package newunits;
import java.math.BigDecimal;
import java.math.MathContext;
import java.util.HashMap;

import fr.cnes.genius.exception.GException;
import fr.cnes.genius.unit.GUnit;

/**
 * Class to define units issued from mols.
 * @author goesterjf
 *
 */
public class MolUnit implements GUnit {
    
    /** Digit precision for BigDecimal */
    private static final MathContext MC = new MathContext(16);
    
    /** Tab for unit names */
    private static final String[] TAB_UNITS_NAME = { "mol", "mmol", "kmol"};
    
    /** Tab for unit conversion factors */
    private static final double[] TAB_UNITS_CONV = { 1., 0.001, 1000.};
    
    /** Map for units and conversion factor in Double */
    private static final HashMap<String, Double> LIST_MOL_UNITS;
    static {
        LIST_MOL_UNITS = new HashMap<String, Double>();
        for (int i = 0; i < TAB_UNITS_CONV.length; i++) {
            LIST_MOL_UNITS.put(TAB_UNITS_NAME[i], TAB_UNITS_CONV[i]); 
        }
    }

    /** Map for units and conversion factor in BigDecimal */
    private static final HashMap<String, BigDecimal> LIST_MOL_UNITS_BD;
    static {
        LIST_MOL_UNITS_BD = new HashMap<String, BigDecimal>();
        for (int i = 0; i < TAB_UNITS_CONV.length; i++) {
            LIST_MOL_UNITS_BD.put(TAB_UNITS_NAME[i], new BigDecimal(TAB_UNITS_CONV[i], MC)); 
        }
    }

    /** Unit name */
    private final String name;
    
    /**
     * Constructor.
     * @param name unit name
     */
    public MolUnit (final String name) {
        
        this.name = name;
    }

    @Override
    public boolean isValidUnitName (String unitName) {
        
        boolean rep = false;
        
        // Loop on the available units
        for (int i = 0; i < TAB_UNITS_NAME.length; i++) {
            if ( unitName.equals(TAB_UNITS_NAME[i])) {
                rep = true;
                break;
            }
        }

        return rep;
        
    }
    
    @Override
    public boolean isCompatible (GUnit unit) {
        
        return isValidUnitName(name);
        
    }

    @Override
    public String getName () {
        return name;
    }

    @Override
    public String getCompactedName () {
        return TAB_UNITS_NAME[0];
    }

    @Override
    public BigDecimal convertToUnit (BigDecimal value, GUnit destUnit) throws GException {
        return destUnit.convertFromMemory(convertToMemory(value));
    }

    @Override
    public double convertToUnit (double value, GUnit destUnit) throws GException {
        return destUnit.convertFromMemory(convertToMemory(value));
    }

    @Override
    public BigDecimal convertToMemory (BigDecimal value) {
        return value.multiply(LIST_MOL_UNITS_BD.get(name), MC);
    }

    @Override
    public double convertToMemory (double value) {
        return value*LIST_MOL_UNITS.get(name);
    }

    @Override
    public BigDecimal convertFromMemory (BigDecimal value) {
        return value.divide(LIST_MOL_UNITS_BD.get(name), MC);
    }

    @Override
    public double convertFromMemory (double value) {
        return value/LIST_MOL_UNITS.get(name);
    }
    
}

