package gantt;

import java.awt.Component;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.AxisLocation;
import org.jfree.chart.labels.IntervalCategoryToolTipGenerator;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.data.category.CategoryDataset;
import org.jfree.data.gantt.Task;
import org.jfree.data.gantt.TaskSeries;
import org.jfree.data.gantt.TaskSeriesCollection;

import fr.cnes.genius.exception.GException;
import fr.cnes.genius.main.GComponent;
import fr.cnes.genius.main.GDisplay;

/**
 * Class to define a Gantt plot.
 * @author goesterjf
 *
 */
public class GFreeChartGantt extends GComponent implements GDisplay {

    /** Back to the next line for Tooltip */
    public static final String RETURN = "<br>";
    
    /** Default delay for tooltip (ms) */
    public static final int DELAY_DEF = 10000;
    
    /** JFreeChart panel */
    private final JFreeChart chart;

    /** JFreeChart panel */
    private final ChartPanel panel;
    
    /** JFreeChart data set */
    private final TaskSeriesCollection dataset;
    
    /** JFreeChart serie */
    private final TaskSeries serie;
    /** sub serie for start dates*/
    private final List<Date> startDates;
    /** sub serie for start dates*/
    private final List<Date> endDates;

    /** Amount of ms for tooltips */
    private int tooltipDelay;
    
    /**
     * Constructor without wiget Id.
     * @param mainTitle     Main title
     * @param xAxisTitle    X axis title
     * @param yAxisTitle    Y axis title
     * @param addTooltips   add tooltips if true
     */
    protected GFreeChartGantt(final String mainTitle, final String xAxisTitle, final String yAxisTitle,
            final boolean addTooltips) {
        this(null, mainTitle, xAxisTitle, yAxisTitle, addTooltips);
    }

    /**
     * Constructor with wiget Id.
     * @param widgetId      widget Id
     * @param mainTitle     Main title
     * @param xAxisTitle    X axis title
     * @param yAxisTitle    Y axis title
     * @param addTooltips   add tooltips if true
     */
    protected GFreeChartGantt(String widgetId,
            final String mainTitle, final String xAxisTitle, final String yAxisTitle,
            final boolean addTooltips) {
        
        super(widgetId);
        
        serie = new TaskSeries(mainTitle);
        startDates = new ArrayList<Date>();
        endDates = new ArrayList<Date>();
        
        dataset = new TaskSeriesCollection();
        dataset.add(serie);

        // Create chart
        chart = ChartFactory.createGanttChart(mainTitle, xAxisTitle, yAxisTitle, dataset, false, addTooltips, false);

        final CategoryPlot categoryPlot = chart.getCategoryPlot();
        categoryPlot.getRenderer().setDefaultToolTipGenerator(new MyToolTipGenerator(
                "{0}, {1}: ", DateFormat.getTimeInstance(DateFormat.SHORT)));

        categoryPlot.setRangeAxisLocation(AxisLocation.BOTTOM_OR_RIGHT);

        panel = new ChartPanel(chart);
        
        // By default tooltip delay
        this.tooltipDelay = DELAY_DEF;
        panel.setDismissDelay(tooltipDelay);

    }
    
    /**
     * Method to add task data.
     * @param nameTask      name of the task
     * @param startDate     start date
     * @param endDate       end date
     */
    public void addTask (final String nameTask, final Date startDate, final Date endDate) {
                
        final Task task = new Task(nameTask, startDate, endDate);
        serie.add(task);
        startDates.add(startDate);
        endDates.add(endDate);

    }
    
    /**
     * Method to be redifined
     * @param col  column number
     * @return String corresponding to the tooltip
     */
    public String getToolTip (final int col) {
        
        final String label = startDates.get(col).toString() + "/" + endDates.get(col).toString();
        
        return label;
        
    }

    /**
     * @return the tooltipDelay
     */
    public int getTooltipDelay() {
        return tooltipDelay;
    }

    /**
     * @param delay the tooltipDelay to set
     */
    public void setTooltipDelay(final int delay) {
        this.tooltipDelay = delay;
    }

    @Override
    public void display() throws GException {
        doDisplay(this);

    }

    @Override
    public Component getComponent() {
        return panel;
    }
    
    /**
     * Private class to redefine the generateToolTip method.
     * @author goesterjf
     */
    private final class MyToolTipGenerator extends IntervalCategoryToolTipGenerator {

        /** Serial Id */
        private static final long serialVersionUID = 1L;

        /**
         * Constructor
         * @param value     JFreechart parameter (not used)
         * @param format    Freechart parameter (not used)
         */
        private MyToolTipGenerator(String value, DateFormat format) {
            super(value, format);
        }

        @Override
        public String generateToolTip(CategoryDataset cds, int row, int col) {
            
            super.generateToolTip(cds, row, col);
            
            return getToolTip(col);
            
        }
        
    }
    
}
