
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Paint;
import java.awt.Shape;
import java.awt.Stroke;
import java.awt.geom.Line2D;
import java.io.UnsupportedEncodingException;

import org.jfree.chart.LegendItem;
import org.jfree.chart.LegendItemCollection;
import org.jfree.chart.LegendItemSource;
import org.jfree.chart.title.LegendTitle;
import org.jfree.data.xy.XYSeries;

import fr.cnes.genius.chart.GDateFormatAbstract;
import fr.cnes.genius.chart.GFreeChartXY;
import fr.cnes.genius.chart.GFreeChartXY.GAxisType;
import fr.cnes.genius.chart.GFreeChartXY.GFreeChartXYItem;
import fr.cnes.genius.chart.GFreeChartXY.GSeriesProperties;
import fr.cnes.genius.chart.GShapeFactory;
import fr.cnes.genius.constraints.GConstraint;
import fr.cnes.genius.exception.GException;
import fr.cnes.genius.exception.GFreeChartException;
import fr.cnes.genius.highLevel.GConsole;
import fr.cnes.genius.highLevel.GEntryReal;
import fr.cnes.genius.highLevel.GMultipleChoice;
import fr.cnes.genius.lowLevel.GFrame;
import fr.cnes.genius.lowLevel.GPanel;
import fr.cnes.genius.main.GEvent;
import fr.cnes.genius.main.GListener;

/**
 * Testing GFreeChartXY with updates and interaction.
 * @author goesterjf
 *
 */
public class TestForGFreeChartXYInteractive extends GPanel implements GListener {

    /** Choice of the plots to interact */
    private final GMultipleChoice choice;
	/** entry for a new abscissa */
    private final GEntryReal newValX;
	/** Command panel */
    private final CommandPanel cmdPan;
    /** Console window */
    private GConsole console;
    /** Plot window */
    private GFreeChartXY plot;
	
	/** Data stored in series */
    private final XYSeries[] dataSeries;

	/**
	 * Constructor.
	 * @throws GFreeChartException exception
	 * @throws UnsupportedEncodingException 
	 */
    public TestForGFreeChartXYInteractive() throws GFreeChartException, UnsupportedEncodingException {
		
	    // Choice widget initialization
        choice = new GMultipleChoice("Choice of the plots:", new int[] {0, 1, 2, 3},
                new String[] {"y=f(x)", "y=f(x^2)", "y=f(x^3)", "Date"});
        choice.setInnerDescendantConstraint(new GConstraint(GConstraint.split(3)), 1);
	    
		// Entry real initialization
        newValX = new GEntryReal("New abscissa to add to the plot:", 0.);
		
        // Global panel for both choice and entry real
        cmdPan = new CommandPanel();
        cmdPan.setConstraint(new GConstraint(GConstraint.newline(), GConstraint.height(70)));
		
        // Console
        console = new GConsole();
        console.redirectSystemStreams(true, true);
        console.setConstraint(new GConstraint(GConstraint.newline(),
                GConstraint.height(100), GConstraint.pushx(), GConstraint.growx()));
        
		// Plot initialization
        final boolean addLegendFlag = true;
        plot = new GFreeChartXY("Dynamic and interactive plots",
                GAxisType.Number, GAxisType.Number, GAxisType.Date, "x", "y", "Date", addLegendFlag);
        //plot = new GFreeChartXY("Dynamic and interactive plots",
        //      GAxisType.Number, GAxisType.Number, GAxisType.Log, "x", "y", "Date", addLegendFlag);
        plot.setConstraint(new GConstraint(GConstraint.newline(), GConstraint.push(), GConstraint.grow(), GConstraint.spanx(99)));
        plot.setInteractive(true);
        plot.setInteractivePrecision(0.01);
        plot.setForcedDrawLine(true);
		
        // Series initialization
        dataSeries = new XYSeries[4];
        final String[] dataSeriesName = {"Y=f(x)", "Y=f(x^2)", "Y=f(x^3)", "Date"};
        final Color[] dataColors = { Color.RED, Color.BLUE, Color.GREEN, Color.BLACK};
        dataSeries[0] = addSeries(dataSeriesName[0], Color.RED, false);
        dataSeries[1] = addSeries(dataSeriesName[1], Color.BLUE, false);
        dataSeries[2] = addSeries(dataSeriesName[2], Color.GREEN, false);
        dataSeries[3] = addSeries(dataSeriesName[3], Color.BLACK, true);
        // Initialization with 5 points
        for (int i = 0; i < 5; i++) {
            addData(i);
        }
        final LegendItemSource source = new LegendItemSource() {
            @Override
            public LegendItemCollection getLegendItems() {
                final LegendItemCollection coll = new LegendItemCollection();
                for (int i = 0; i < dataSeriesName.length; i++) {
                    // Getting the color of the serie
                    final Paint fillPaint = dataColors[i];
                    final Stroke stroke = new BasicStroke();
                    final double width = 20.;
                    final double posy = 3.;
                    final Shape shape = new Line2D.Double(0., posy, width, posy);
                    final LegendItem item = new LegendItem(dataSeriesName[i],"desc","toolTip","url", shape, stroke, fillPaint);
                    coll.add(item);
                }
                return coll;
            }
        };
        plot.addLegend(new LegendTitle(source));
        //plot.removeLegend();

    }
	
    /**
     * Adding series
     * @param title title of the serie
     * @param color color of the serie
     * @param isSecondaryAxis on secondary axis if true
     * @return the XY series
     * @throws GFreeChartException occured when adding a serie.
     */
    private XYSeries addSeries (final String title, final Color color, final boolean isSecondaryAxis) throws GFreeChartException {
        
        final XYSeries serie = new XYSeries(title, true);
        final GSeriesProperties prop = new GSeriesProperties(color, GShapeFactory.createDiamond(3), null, null, null);
        plot.addSerie(isSecondaryAxis, prop, serie);
        
        return plot.getXYDataSet(title, isSecondaryAxis);
        
    }

    /**
	 * Adding a data to the list.
	 * @param val	value to add.
	 */
    private void addData ( final double val ) {
        final int[] selVals = choice.getSelectedValues();
        for (int i = 0; i < selVals.length; i++) {
            if ( selVals[i] == 3 ) {
                dataSeries[selVals[i]].add(val, 1000*val*GDateFormatAbstract.DAYS);
                //dataSeries[selVals[i]].add(val, Math.pow(10, val));
            } else {
                dataSeries[selVals[i]].add(val, Math.pow(val, 1+selVals[i]));
            }
        }
    }
	
    @Override
    public void generic() { }

    @Override
    public void display() throws GException {
        put(cmdPan);
        put(console);
        put(plot);
    }
	
    @Override
	public void before(GEvent arg0) throws GException {
		// Nothing to do ...
    }

    @Override
    public void after(GEvent arg0) throws GException {
        
        if ( arg0.contains(newValX)) {
            addData(newValX.getValue());
        }
        
        if ( arg0.contains(plot) ) {
            final GFreeChartXYItem coordsAbsMain = plot.getPlaneCoordinates(false);
            System.out.println("Absolute coordinates: "+coordsAbsMain.getStrX()+" / "+coordsAbsMain.getStrY());
            final GFreeChartXYItem coordsAbsMainSecond = plot.getPlaneCoordinates(true);
            System.out.println("Relative coordinates: "+coordsAbsMainSecond.getStrX()+" / "+coordsAbsMainSecond.getStrY());
            final GFreeChartXYItem coordsMain = plot.getItemCoordinates();
            if ( coordsMain != null ) {
                System.out.println("Serie: "+coordsMain.getSeriesName());
                System.out.println("Index in the serie: "+coordsMain.getIndex());
                System.out.println("Position: "+coordsMain.getStrX()+" / "+coordsMain.getStrY());
            } else {
                System.out.println("Too far from a point ...");
            }
        }
        
    }
    
    /**
     * Private class to set both widgets in a single panel.
     * @author goesterjf
     *
     */
    private class CommandPanel extends GPanel {

        @Override
        public void display() throws GException {
            put(choice);
            put(newValX);
        }

        @Override
        public void generic() throws GException {
        }
        
    }
    
	/**
	 * @param args arguments (none)
	 * @throws GFreeChartException GENIUS exception
	 * @throws UnsupportedEncodingException GENIUS exception
	 */
    public static void main(String[] args) throws GFreeChartException, UnsupportedEncodingException {
        final TestForGFreeChartXYInteractive pan = new TestForGFreeChartXYInteractive();
        final GFrame frame = new GFrame("Plots", pan);	
        frame.display();
    }

}
