import java.awt.Color;

import org.jfree.data.xy.XYSeries;

import fr.cnes.genius.chart.GFreeChartXY;
import fr.cnes.genius.chart.GFreeChartXY.GSeriesProperties;
import fr.cnes.genius.chart.GShapeFactory;
import fr.cnes.genius.constraints.GConstraint;
import fr.cnes.genius.constraints.GConstraint.YAlign;
import fr.cnes.genius.exception.GArithmeticException;
import fr.cnes.genius.exception.GException;
import fr.cnes.genius.exception.GFreeChartException;
import fr.cnes.genius.highLevel.GSliderRealWithLabel;
import fr.cnes.genius.highLevel.GSliderWithLabel;
import fr.cnes.genius.lowLevel.GButton;
import fr.cnes.genius.lowLevel.GFrame;
import fr.cnes.genius.lowLevel.GIcon;
import fr.cnes.genius.lowLevel.GPanel;
import fr.cnes.genius.main.GEvent;
import fr.cnes.genius.main.GListener;

/**
 * Testing GFreeChartXY with animation.
 * @author goesterjf
 *
 */
public class TestForGFreeChartXYAnimation extends GPanel implements GListener {

    /** Maximum amount of milliseconds between two points */
    private static final int MAX_MILLIS = 1000;
    /** Pause label */
    private static final String START = "Start";
    /** Resume label */
    private static final String STOP = "Stop";
    /** Pause label */
    private static final String PAUSE = "Pause";
    /** Resume label */
    private static final String RESUME = "Resume";
    /** Sliders positionning */
    private static final String HORIZONTAL = "HORIZONTAL";
    
    /** Command panel */
    private final CommandPanel cmdPan;
    /** Start button */
    private final GButton startStopBut;
    /** Pause button */
    private final GButton pauseBut;
    /** Slider for animation velocity */
    private final GSliderWithLabel sliderAcc;
    /** Slider for animation position */
    private final GSliderRealWithLabel sliderPos;
    
    /** Plot window */
    private final GFreeChartXY plot;
    
    /** True if the animation is stopped */
    private boolean isStop;
    /** True if the animation is in pause */
    private boolean isPause;
    /** True if a position command occurred */
    private boolean isPosition;
    /** Animation class */
    private Animation animation;
    /** Plot properties */
    private final GSeriesProperties prop;

    /** Data stored in series */
    private XYSeries dataSeries;
    
    /** Abcissae data */
    private final double[] tabAbs;
    /** Ordinates data */
    private final double[] tabOrd;
    /** Amount of points */
    private final int nbpts;
    /** Current commanded point */
    private double curCmdPoint;
    
    /**
     * Constructor.
     * @param tabAbs Abcissae data
     * @param tabOrd Ordinates data
     * @throws GFreeChartException GENIUS exception
     */
    public TestForGFreeChartXYAnimation(final double[] tabAbs, final double[] tabOrd) throws GFreeChartException {
        
        this.tabAbs = tabAbs;
        this.tabOrd = tabOrd;
        this.nbpts = tabAbs.length;
        
        isStop = true;
        isPause = false;
        isPosition = false;
        curCmdPoint = 0;
        
        startStopBut = new GButton(START);
        startStopBut.setIcon(new GIcon(GIcon.Type.START, 12));
        startStopBut.setConstraint(new GConstraint(GConstraint.newline(true), GConstraint.split(2),
                GConstraint.spanx(99), GConstraint.gapbottom(20)));
        pauseBut = new GButton(PAUSE);
        pauseBut.setConstraint(new GConstraint(GConstraint.newline(false)));
        pauseBut.setEnabled(false);

        sliderAcc = new GSliderWithLabel("Acceleration:", HORIZONTAL, 1, 10, 5, 1);
        sliderAcc.setInnerDescendantConstraint(new GConstraint(GConstraint.newline(true),
                GConstraint.split(2), GConstraint.aligny(YAlign.Top), GConstraint.width(22)), 0, 0);
        sliderAcc.setInnerDescendantConstraint(new GConstraint(GConstraint.newline(false), GConstraint.aligny(YAlign.Top)), 0, 1);
        sliderAcc.setInnerDescendantConstraint(new GConstraint(GConstraint.newline(false), GConstraint.aligny(YAlign.Top)), 1);
        sliderAcc.setInnerDescendantConstraint(new GConstraint(GConstraint.newline(false), GConstraint.width(300)), 2);
        
        final int nbPos = (int)((tabAbs[nbpts-1]-tabAbs[0])/5);
        sliderPos = new GSliderRealWithLabel("Position:", HORIZONTAL, tabAbs[0], tabAbs[nbpts-1], tabAbs[0], nbPos, 1);
        sliderPos.setInnerDescendantConstraint(new GConstraint(GConstraint.newline(true),
                GConstraint.split(2), GConstraint.aligny(YAlign.Top)), 0, 0);
        sliderPos.setInnerDescendantConstraint(new GConstraint(GConstraint.newline(false), GConstraint.aligny(YAlign.Top)), 0, 1);
        sliderPos.setInnerDescendantConstraint(new GConstraint(GConstraint.newline(false), GConstraint.aligny(YAlign.Top)), 1);
        sliderPos.setInnerDescendantConstraint(new GConstraint(GConstraint.newline(false), GConstraint.width(300)), 2);
        sliderPos.setEnabled(false);
        
        cmdPan = new CommandPanel();
                
        // Plot initialization
        final boolean addLegendFlag = true;
        plot = new GFreeChartXY("y = x^2", "x", "y", null, addLegendFlag);
        plot.setConstraint(new GConstraint(GConstraint.newline(), GConstraint.push(),
                GConstraint.grow(), GConstraint.split(99), GConstraint.spanx(99)));
        plot.setForcedDrawLine(true);
        plot.setXZoom(tabAbs[0], tabAbs[nbpts-1]);
        plot.setYZoom(false, tabOrd[0], tabOrd[nbpts-1]);
        
        prop = new GSeriesProperties(Color.RED, GShapeFactory.createDiamond(3), null, null, null);
        
    }
    
    @Override
    public void generic() { }

    @Override
    public void display() throws GException {
        put(cmdPan);
        put(plot);
    }
    
    @Override
    public void before(GEvent arg0) throws GException {
        // Nothing to do ...
    }

    @Override
    public void after(GEvent arg0) throws GException {
        
        if ( arg0.contains(startStopBut) ) {
            
            if ( isStop ) {
                // We start the animation
                startStopBut.setText(START);
                startStopBut.setIcon(new GIcon(GIcon.Type.STOP, 12));
                pauseBut.setEnabled(true);
                sliderPos.setEnabled(false);
                isStop = false;
                dataSeries = new XYSeries("Square", true);
                dataSeries.add(tabAbs[0], tabOrd[0]);
                plot.removeAllSeries();
                plot.addSerie(false, prop, dataSeries);
                animation = new Animation();
                animation.start();
            } else {
                // We stop the animation
                startStopBut.setText(STOP);
                startStopBut.setIcon(new GIcon(GIcon.Type.START, 12));
                pauseBut.setText(PAUSE);
                pauseBut.setEnabled(false);
                isStop = true;
                isPause = false;
            }
        }
        
        if ( arg0.contains(pauseBut) ) {
            
            if ( isPause ) {
                // We resume the animation
                isPause = false;
                pauseBut.setText(PAUSE);
                sliderPos.setEnabled(false);
            } else {
                // We suspend the animation
                isPause = true;
                pauseBut.setText(RESUME);
                sliderPos.setEnabled(true);
            }
            
        }
        
        if ( arg0.contains(sliderPos) ) {
            
            curCmdPoint = sliderPos.getValue();
            isPosition = true;
            
        }
        
    }
    
    /**
     * Private class to define the command sub panel (useful to manage easily widgets positions).
     * @author goesterjf
     *
     */
    private class CommandPanel extends GPanel {
        
        public CommandPanel() {
        }

        @Override
        public void display () throws GException {
            generic();
        }

        @Override
        public void generic () throws GException {
            put(startStopBut);
            put(pauseBut);
            put(sliderAcc);
            put(sliderPos);
        }
        
    }
    
    /**
     * Private class to manage animation.
     * @author goesterjf
     *
     */
    private class Animation extends Thread {
        
        public void run() {
            
            // Loop on the data
            
            for (int i = 0; i < nbpts; i++) {
                
                if ( isStop == true ) {
                    break;
                }
                
                if ( isPause ) {
                    do {
                        try {
                            Thread.sleep(100);
                        } catch ( InterruptedException e ) {
                            // Nothing to do ...
                        }
                    } while ( isPause );
                }
                
                if ( isPosition ) {
                    // We change the position of the current point
                    if ( curCmdPoint < tabAbs[i] ) {
                        // Backward ...
                        for (int j = i; j >= 0; j--) {
                            if ( curCmdPoint < tabAbs[j] ) {
                                dataSeries.remove(j);
                            } else {
                                i = j;
                                isPosition = false;
                                break;
                            }
                        }
                    } else {
                        // Forward
                        for (int j = i; j < tabAbs.length; j++) {
                            if ( curCmdPoint > tabAbs[j] ) {
                                dataSeries.add(tabAbs[j], tabOrd[j]);
                            } else {
                                i = j;
                                isPosition = false;
                                break;
                            }
                        }
                    }
                }
                
                dataSeries.add(tabAbs[i], tabOrd[i]);
                try {
                    sliderPos.setValue(tabAbs[i]);
                } catch ( GArithmeticException e1 ) {
                    // Nothing to do ...
                }
                try {
                    final long lll = (int)MAX_MILLIS/sliderAcc.getValue();
                    Thread.sleep(lll);
                } catch ( InterruptedException e ) {
                    // Nothing to do ...
                }
                
            }
            
            isStop = true;
            startStopBut.setIcon(new GIcon(GIcon.Type.START, 12));
            pauseBut.setEnabled(false);

        }
    }

    /**
     * @param args
     * @throws GFreeChartException 
     */
    public static void main(String[] args) throws GFreeChartException {
        
        final int npts = 50;
        final double[] tabAbs;
        final double[] tabOrd;

        tabAbs = new double[npts];
        tabOrd = new double[npts];
        for (int i = 0; i < npts; i++) {
            tabAbs[i] = i;
            tabOrd[i] = i*i;
        }
        
        final TestForGFreeChartXYAnimation pan = new TestForGFreeChartXYAnimation(tabAbs, tabOrd);
        final GFrame frame = new GFrame("Plots", pan);    
        frame.display();
    }

}
