package GStandardApplication.ihm;

import GStandardApplication.calc.Vehicle;
import fr.cnes.genius.exception.GException;
import fr.cnes.genius.exception.GIntervalException;
import fr.cnes.genius.highLevel.GEntryInt;
import fr.cnes.genius.highLevel.GEntryReal;
import fr.cnes.genius.interval.GInterval;
import fr.cnes.genius.interval.GIntervalAbstract.Rule;
import fr.cnes.genius.lowLevel.GPanel;
import fr.cnes.genius.main.GClear;
import fr.cnes.genius.main.GCondensedStatus;
import fr.cnes.genius.main.GCondensedStatusInterface;
import fr.cnes.genius.main.GReadWrite;
import fr.cnes.genius.unit.GMetricUnit;
import fr.cnes.genius.unit.GUnit;

/**
 * Class managing vehicle data.
 * @author goesterjf
 *
 */
public class WidVehicle extends GPanel implements GReadWrite, GClear, GCondensedStatusInterface  {
    
    /** List of units for mass */
    public static final GUnit[] UNITS_MASS = {new GMetricUnit("kg"), new GMetricUnit("kt")};
    /** List of units for surface */
    public static final GUnit[] UNITS_SURFACE = {new GMetricUnit("m^2")};
    
    /** Widget for dry mass entry */
    private final GEntryReal dryMass;
    /** Widget for ergols mass entry */
    private final GEntryReal ergMass;
    /** Widget for reference area entry */
    private final GEntryReal sref;
    /** Widget for amount of iterations */
    private final GEntryInt nbIter;
    /**
     * Constructor.
     */
    public WidVehicle () {
        
        super();
        
        dryMass = new GEntryReal("Dry mass:", 0., UNITS_MASS);
        dryMass.setNameInConfigFile("dryMass");
        try {
            dryMass.addGInterval(new GInterval(0., Double.POSITIVE_INFINITY,
                                               Rule.EXCLUSIVE, Rule.INCLUSIVE));
        } catch (GIntervalException err) {
        	System.out.println("problem in the dry mass interval of validity ...");
        }
        
        ergMass = new GEntryReal("Ergol mass:", 0., UNITS_MASS);
        ergMass.setNameInConfigFile("ergMass");
        try {
            ergMass.addGInterval(new GInterval(0., Double.POSITIVE_INFINITY, Rule.EXCLUSIVE, Rule.INCLUSIVE));
        } catch (GIntervalException err) {
        	System.out.println("problem in the ergol mass interval of validity ...");
        }
        
        sref = new GEntryReal("Reference surface:", 0., UNITS_SURFACE);
        sref.setNameInConfigFile("sref");
        try {
            sref.addGInterval(new GInterval(0., Double.POSITIVE_INFINITY,
                                            Rule.EXCLUSIVE, Rule.INCLUSIVE,
                                            0., Double.POSITIVE_INFINITY,
                                            Rule.INCLUSIVE, Rule.INCLUSIVE));
        } catch (GIntervalException err) {
        	System.out.println("problem in the reference surface interval of validity ...");
        }
        
        nbIter = new GEntryInt("Amount of iterations:", 100);
        try {
            nbIter.addGInterval(new GInterval(0., Double.POSITIVE_INFINITY, Rule.EXCLUSIVE, Rule.INCLUSIVE));
        } catch ( GIntervalException e ) {
            System.out.println("problem in the amount of iteration validity ...");
        }
        
    }
    
    /**
     * Method managing generic functionalities.
     * @throws GException  GENIUS exception
     */
    @Override
    public void generic() throws GException {
        beginOfElement("Vehicle", "vehicle");
        put(dryMass);
        put(ergMass);
        put(sref);
        put(nbIter);
        endOfElement();
    }

    /**
     * Method managing display.
     * @throws GException  GENIUS exception
     */
    public void display() throws GException {
        generic();
    }

    /**
     * Method managing the way to clear data.
     * @throws GException  GENIUS exception
     */
    public void clear() throws GException {
        generic();
    }

    /**
     * Method managing the way to read in a file.
     * @throws GException  GENIUS exception
     */
    public void read() throws GException {
        generic();
    }

    /**
     * Method managing the way to write in a file.
     * @throws GException  GENIUS exception
     */
    public void write() throws GException {
        generic();
    }

    /**
     * Building the Vehicle object.
     * @return object containing all the data needed for vehicle data.
     */
    public Vehicle getCalcObject() {
        return  new Vehicle(dryMass.getValue(), ergMass.getValue(), sref.getValue(), nbIter.getValue());
    }
    
    @Override
    /**
     * Defining which sub-widgets are concerned by the condensed status management.
     */
    public void updateCondensedStatus(GCondensedStatus arg0) {
        // dryMass, ergMass and sref are checked
        arg0.update(dryMass, ergMass, sref, nbIter);	
    }
}
